/** \file PFConfig.h
  */


/*!
 * \mainpage PFConfig DLL Documentation
 * \version 1.0.2
 * \author Photonfocus AG
 * \date 01/2010
 *
 * \section intro   Introduction
 * The software package that comes with a Photonfocus camera contains an 
 * application and a SDK to evaluate the camera. With PFRemote, all camera 
 * parameters can be configured. Next to the pflib SDK for C++ application 
 * development there is also a PFConfig DLL for an independent integration
 * in several 3rd party application like Labview, Mathlab, VB, C#, etc.
 *
 * The PFConfig DLL is a software development kit as well as for the  
 * Photonfocus USB camera as for the camealink camera. With the PFConfig, 
 * all camera parameters such as exposure time, ROI, etc. can be configured.
 * The PFConfig DLL is based on the pflib library and also on the fglib2.2.lib 
 * display_lib.lib for USB cameras.
 *
 * Most of the functions return an error code in the case the function fails.
 * Please refer to the documentation of the pflib for a detailed description
 * of the error codes. You can also call the function pfGetErrorMsg() which
 * returns the latest error message. This function should be called if the
 * return code of any function is unlike 0.
 * 
 * 
 * There are examples available about how to use the PFConfig DLL.
 *
 *
 */

/**
 * \example PFConfig_Example.vbproj
 * VB 2008 - Example shows how to configure (read/write) the camera.
 */

/**
 * \example PFConfig_Example.vcproj
 * VC 2008 - Example shows how to configure (read/write) the camera.
 */

#ifdef PFCONFIG_EXPORTS
#ifdef __cplusplus
extern "C"
#define PFCONFIG_API  extern "C" __declspec(dllexport)
#else
#define PFCONFIG_API __declspec(dllexport)
#endif

#else
#ifdef __cplusplus
#define PFCONFIG_API extern "C" __declspec(dllimport)
#else
#define PFCONFIG_API __declspec(dllimport)
#endif
#endif

#include <stdio.h>
#include <string>
using namespace std;

///
/// \brief Get the version of the PFConfig Dll.
/// \param[in] iPort Port number [0...(NumPort-1)].
///
/// \return A pointer to a array of character containing a version.
///
PFCONFIG_API char* pfGetVersion(int iPort);


///
/// \brief Init of the USB camera frame grabber with a camera-dll.
///	This function is only needed when using a USB camera. 
/// \param[in] pDllName	The name of the camera-dll, (i,e mv_d640_48_u2.dll, mv_d1024e_40_u2.dll)
///
/// \return Error code or 0 if DLL successfully loaded.
///
PFCONFIG_API int pfInitUsb(char* pDllName);


///
/// \brief Returns The number of available ports.
///
/// \return The number of available ports or error code.
///
PFCONFIG_API int pfGetNumPort(void);


///
/// \brief Returns the current port information of the given port. 
///	The port information is returned in the following structure:
///	"Portnumber, Manufactor, Portname, Version, Type". The  
///	parameters are summarized in a string and separated by a comma.
/// \param[in] iPort Port number [0...(NumPort-1)].
///
/// \return A pointer to a string containing the port description.
///
PFCONFIG_API const char* pfGetPortInfo(int iPort);


///
/// \brief Returns the current property description by the given token ID.  
/// A full list with all properties will be created when the pfOpenPort()  
/// gets called. Use the function pfGetNumProperties() to get the total 
/// number of available properties made available from the connected camera.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[out] iProperty Property token ID [0...(NumProperties-1)].
/// \param[out] iPropType The type of the property.
/// Following types are supported by Library
/// 2 : Integer
/// 3 : Float
/// 4 : Bool
/// 5 : Mode
/// 7 : String
/// 8 : Command

/// \param[in] iPropFlag The property flags.
/// These flags are queried via pfProperty_GetFlags().
/// 0 : Readable/Writeable
/// 2 : Property is private
/// 4 : Big endian, if Register node (not used)
/// 16 : Readonly
/// 32 : Writeonly
/// 64 : Property is currently inactive

///
/// \return A pointer to a string with the property description from the given ID.
///
PFCONFIG_API const char* pfGetPropertyInfo(int iPort, int iProperty, int* iPropType, int* iPropFlag);


///
/// \brief Returns The number of available camera properties.
/// \param[in] iPort Port number [0...(NumPort-1)].
///
/// \return The number of available camera properties or error code.
///
PFCONFIG_API int pfGetNumProperties(int iPort);


///
/// \brief Returns the latest error message. This function should be called
/// if the return code of any function is unlike 0. Please refer to the pflib
/// documentation for a detailed description of all error codes.
/// \param[in] iPort Port number [0...(NumPort-1)].
///
/// \return Pointer to the latest error message.
///
PFCONFIG_API const char* pfGetErrorMsg(int iPort);


///
/// \brief Close the camera port if there is any open port.
/// \param[in] iPort Port number [0...(NumPort-1)].
///
/// \return Error code or 0 if port successfully closed.
///
PFCONFIG_API int pfClosePort(int iPort);


///
/// \brief Open the camera port from the given port ID.
///
/// \param[in] iPort The camera port number to open for communicaton.
/// \param[in] iCheckHigherBaudrate Check if the camera supports a higher baudrate than 9600.
///  If the camera supports a higher baudrate switch to 57600bps.
/// \return Error code or 0 if port successfully opened.
///
PFCONFIG_API int pfOpenPort(int iPort, int iCheckHigherBaudrate=false);


///
/// \brief Sets the given camera property value.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[in] pProperty The camera property name to set.
/// \param[in] pValue The string value to set.
///
/// \return Return 0 if property has been successfully write.
///			Otherwise return Error code.		
///
PFCONFIG_API int pfSetCameraPropertyS(int iPort, const char* pProperty, const char* pValue);


///
/// \brief Gets the given camera property value.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[in] pProperty The camera property name to read.
/// \param[in] pValue The string (pointer) to save the property value.
/// \param[in] iSize The maximum memory size for the property to save.
///
/// \return Return 0 if property has been successfully read.
///			Otherwise return Error code.
///
PFCONFIG_API int pfGetCameraPropertyS(int iPort, const char* pProperty, char* pValue, int iSize);


///
/// \brief Sets the given camera property value.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[in] pProperty The camera property name to set.
/// \param[in] fValue The floating value to set.
///
/// \return Return 0 if property has been successfully write.
///			Otherwise return Error code.
///
PFCONFIG_API int pfSetCameraPropertyF(int iPort, const char* pProperty, float fValue);


///
/// \brief Gets the given camera property value.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[in] pProperty The camera property to set.
///
/// \return Property floating value if property has been successfully read.
///			Otherwise return Error code.
///
PFCONFIG_API float pfGetCameraPropertyF(int iPort, const char* pProperty);


///
/// \brief Gets the given camera property value.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[in] pProperty The camera property to set.
/// \param[in] fValue Property floating value.
///
/// \return Return 0 if property has been successfully read.
///			Otherwise return Error code.
///
PFCONFIG_API int pfGetCameraPropertyFloat(int iPort, const char* pProperty, float *fValue);


///
/// \brief Sets the given camera property value.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[in] pProperty The camera property name to set.
/// \param[in] iValue The integer value to set.
///
/// \return Return 0 if property has been successfully write.
///			Otherwise return Error code.
///
PFCONFIG_API int pfSetCameraPropertyI(int iPort, const char* pProperty, int iValue);


///
/// \brief Gets the given camera property value.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[in] pProperty The camera property name to set.
///
/// \return Property integer value if property has been successfully read.
///			Otherwise return Error code.
///
PFCONFIG_API int pfGetCameraPropertyI(int iPort, const char *pProperty);


///
/// \brief Gets the given camera property value.
/// \param[in] iPort Port number [0...(NumPort-1)].
/// \param[in] pProperty The camera property name to set.
/// \param[in] iValue Property integer value.
///
/// \return Return 0 if property has been successfully read.
///			Otherwise return Error code.
///
PFCONFIG_API int pfGetCameraPropertyInt(int iPort, const char *pProperty, int *iValue);
