// comserver.c
// Compiler independent interface to the framegrabber serial port
//

#include <fcntl.h>
#include <stdio.h>
#include <inttypes.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <signal.h>
#include <errno.h>

#include <clser.h>

// Enable debug output
//#define debug

#define DELAY 10000		// us

#define socket_name "/tmp/pfcomserver.socket"

int main(int argc, char * argv[]){
    int res;
    
    signal (SIGPIPE, SIG_IGN);

    int sock = socket (PF_UNIX, SOCK_STREAM, 0);
    if (sock < 0) perror("socket");

    unlink(socket_name);

    // Give the socket a name.
    struct sockaddr_un name;
    memset(&name, 0, sizeof(name));
    name.sun_family = AF_UNIX;
    strncpy(name.sun_path,socket_name,sizeof(name.sun_path));
    name.sun_path[sizeof(name.sun_path)-1]=0;

    if (bind (sock, (struct sockaddr *) &name, sizeof (name)) < 0) perror ("bind");
    if (listen (sock, 10) < 0) perror ("listen");

    chmod(socket_name, 0666);

    // Add to select set?!
    fd_set active_fd_set;
    fd_set select_fd_set;

    FD_ZERO (&active_fd_set);
    FD_SET (sock, &active_fd_set);

    // Open frame grabber serial port
    int serialIndex = 0;
    int time_max = 100;	// in ms
    void *handle;
    unsigned int dwLength;

    clSerialInit(serialIndex, (void **) &handle);

    // Server loop
    struct sockaddr_un clientname;
    int size = sizeof (clientname);
    int sockNew;
    char msg[2048];

    uint32_t magic;

    while (1) {
        select_fd_set = active_fd_set;
        res = select (sock + 1, &select_fd_set, NULL, NULL, NULL);
        if (res <= 0) continue;

        if ((sockNew = accept (sock, (struct sockaddr *) &clientname, &size)) < 0) continue;

        while (1) {
            res = read(sockNew, &magic, 4);
            if (res != 4) {
                if (res) {
		    fprintf(stderr, "Read magic from socket is failed (res = %i), errno: %i\n", res, errno);
		}
                close(sockNew);
                break;
            }

//            printf("%i\n", magic);

            if (magic) {
		dwLength = magic;
		
#ifdef debug
                printf("clSerialRead -- Request %d bytes\n", dwLength);
#endif

                // Read data from the camera
                res = clSerialRead((void*)handle, (char*)msg, &dwLength, time_max);
                if (res < 0) {
			// Thats normal and could happen
#ifdef debug
                    fprintf(stderr, "Read from camera is failed, res: %i, errno: %i\n", res, errno);
#endif
		    
		    magic = 0xFFFFFFFF;
		    res = write(sockNew, &magic, 4);
		    if (res != 4) {
            		fprintf(stderr, "Write magic to socket is failed, errno: %i\n", errno);
                	close(sockNew);
                	break;
		    }

//                    close(sockNew);
                    continue;
                } else {
		    magic = 0;
		    res = write(sockNew, &magic, 4);
		    if (res != 4) {
            		fprintf(stderr, "Write magic to socket is failed, errno: %i\n", errno);
                	close(sockNew);
                	break;
		    }
		}
		
		usleep(DELAY);
		

                // Send result back to the pipe
                res = write(sockNew, msg, dwLength);
                if (res != dwLength) { 
                    fprintf(stderr, "Write to socket is failed, errno: %i\n", errno);
                    close(sockNew);
                    break;
                }
	    } else {
                res = read(sockNew, msg, sizeof(msg));
                if (res <= 0) {
                    fprintf(stderr, "Read from socket is failed, errno: %i\n", errno);
                    close(sockNew);
                    break;
                }
#ifdef debug
                printf("clSerialWrite -- Send %d bytes\n", res);
#endif
	    
                dwLength = res;

                res = clSerialWrite((void *)handle, (char*)msg, &dwLength, time_max);
                if (res < 0) {
			// We can also just report error back
                    fprintf(stderr, "Write to camera is failed, res: %i, errno: %i\n", res, errno);
                    close(sockNew);
                    continue;
                }
            } 
        }


        close(sockNew);

    }

    return(0);
}
